const Database = require('./db');

class Auth {
  constructor() {
    this.db = new Database();
  }

  hashPassword(password) {
    // For now, keep plain text (matching current PHP implementation)
    // In production, use bcrypt: return bcrypt.hashSync(password, 10);
    return password;
  }

  comparePassword(password, storedPassword) {
    // Plain text comparison (matching current PHP implementation)
    // In production, use bcrypt: return bcrypt.compareSync(password, storedPassword);
    return password === storedPassword;
  }

  async registerUser(fullName, username, email, password) {
    // Check if user exists by username
    if (username) {
      const existingUser = await this.db.getUserByUsername(username);
      if (existingUser) {
        throw new Error('User with this username already exists');
      }
    }

    // Check if user exists by email
    const existingUser = await this.db.getUserByEmail(email);
    if (existingUser) {
      throw new Error('User with this email already exists');
    }

    // Store password as plain text (matching current PHP implementation)
    const plainPassword = password;

    // Create user
    const userId = await this.db.createUser(fullName, username, email, plainPassword);

    // Return user without password
    const user = await this.db.getUserById(userId);
    if (user) {
      delete user.password;
    }
    return user;
  }

  async authenticateUser(username, password) {
    const user = await this.db.getUserByUsername(username);

    if (!user) {
      return null;
    }

    const isMatch = this.comparePassword(password, user.password);

    if (!isMatch) {
      return null;
    }

    // Return user without password
    const userWithoutPassword = { ...user };
    delete userWithoutPassword.password;
    return userWithoutPassword;
  }

  async getUserDashboard(userId) {
    try {
      const user = await this.db.getUserById(userId);

      if (!user) {
        throw new Error(`User not found with ID: ${userId}`);
      }

      // Remove password from user data
      const userWithoutPassword = { ...user };
      delete userWithoutPassword.password;

      const accounts = await this.db.getUserAccounts(userId);
      const totalBalance = await this.db.getTotalBalance(userId);
      const transfers = await this.db.getUserTransfers(userId, 10);

      return {
        user: {
          id: userWithoutPassword.id || 0,
          full_name: userWithoutPassword.full_name || 'User',
          username: userWithoutPassword.username || '',
          email: userWithoutPassword.email || ''
        },
        accounts: Array.isArray(accounts) ? accounts : [],
        totalBalance: totalBalance ? parseFloat(totalBalance) : 0.0,
        recentTransfers: Array.isArray(transfers) ? transfers : [],
        transfers: Array.isArray(transfers) ? transfers : [] // Also include for compatibility
      };
    } catch (error) {
      if (error.message.includes('User not found')) {
        throw error;
      }
      throw new Error(`Error getting dashboard data: ${error.message}`);
    }
  }
}

module.exports = Auth;

