const { Pool } = require('pg');
const { dbConfig } = require('./config');

class Database {
  constructor() {
    // Create connection pool
    this.pool = new Pool(dbConfig.connectionString ? {
      connectionString: dbConfig.connectionString,
      ...dbConfig,
    } : dbConfig);
    
    // Initialize database on first connection
    this.initialized = false;
  }

  async initialize() {
    if (this.initialized) return;
    
    try {
      await this.initializeDatabasePostgres();
      this.initialized = true;
    } catch (error) {
      throw new Error(`PostgreSQL initialization failed: ${error.message}`);
    }
  }

  async initializeDatabasePostgres() {
    const client = await this.pool.connect();
    try {
      // Users table
      await client.query(`
        CREATE TABLE IF NOT EXISTS users (
          id SERIAL PRIMARY KEY,
          full_name VARCHAR(255) NOT NULL,
          username VARCHAR(100) UNIQUE,
          email VARCHAR(255) UNIQUE NOT NULL,
          password VARCHAR(255) NOT NULL,
          total_balance DECIMAL(15,2) DEFAULT 0,
          currency VARCHAR(10) DEFAULT 'AUD',
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
      `);

      // Create indexes for users table
      await client.query('CREATE INDEX IF NOT EXISTS idx_users_email ON users(email)');
      await client.query('CREATE INDEX IF NOT EXISTS idx_users_username ON users(username)');

      // Accounts table
      await client.query(`
        CREATE TABLE IF NOT EXISTS accounts (
          id SERIAL PRIMARY KEY,
          user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
          currency VARCHAR(10) NOT NULL,
          balance DECIMAL(15,2) DEFAULT 0,
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          UNIQUE(user_id, currency)
        )
      `);

      // Create indexes for accounts table
      await client.query('CREATE INDEX IF NOT EXISTS idx_accounts_user_id ON accounts(user_id)');
      await client.query('CREATE INDEX IF NOT EXISTS idx_accounts_currency ON accounts(currency)');

      // Transfers table
      await client.query(`
        CREATE TABLE IF NOT EXISTS transfers (
          id SERIAL PRIMARY KEY,
          user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
          from_currency VARCHAR(10),
          to_currency VARCHAR(10),
          send_amount DECIMAL(15,2),
          fee DECIMAL(15,2),
          exchange_rate DECIMAL(20,6),
          receive_amount DECIMAL(15,2),
          status VARCHAR(50) DEFAULT 'PENDING',
          recipient_name VARCHAR(255),
          recipient_account VARCHAR(255),
          recipient_bank VARCHAR(255),
          recipient_country VARCHAR(100),
          rate_locked_at TIMESTAMP,
          rate_expires_at TIMESTAMP,
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
      `);

      // Create indexes for transfers table
      await client.query('CREATE INDEX IF NOT EXISTS idx_transfers_user_id ON transfers(user_id)');
      await client.query('CREATE INDEX IF NOT EXISTS idx_transfers_status ON transfers(status)');
      await client.query('CREATE INDEX IF NOT EXISTS idx_transfers_created_at ON transfers(created_at)');

      // Sessions table for database-backed sessions
      await client.query(`
        CREATE TABLE IF NOT EXISTS sessions (
          session_id VARCHAR(255) PRIMARY KEY,
          user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
          transfer_data JSONB,
          expires_at TIMESTAMP,
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
      `);

      // Create indexes for sessions table
      await client.query('CREATE INDEX IF NOT EXISTS idx_sessions_user_id ON sessions(user_id)');
      await client.query('CREATE INDEX IF NOT EXISTS idx_sessions_expires_at ON sessions(expires_at)');
    } finally {
      client.release();
    }
  }

  async getUserByEmail(email) {
    await this.initialize();
    const result = await this.pool.query('SELECT * FROM users WHERE email = $1', [email]);
    return result.rows[0] || null;
  }

  async getUserByUsername(username) {
    await this.initialize();
    const result = await this.pool.query('SELECT * FROM users WHERE username = $1', [username]);
    return result.rows[0] || null;
  }

  async getUserById(id) {
    await this.initialize();
    const result = await this.pool.query('SELECT * FROM users WHERE id = $1', [id]);
    return result.rows[0] || null;
  }

  async createUser(fullName, username, email, hashedPassword) {
    await this.initialize();
    const client = await this.pool.connect();
    try {
      await client.query('BEGIN');
      
      // Create user with initial balance of 0
      const userResult = await client.query(
        'INSERT INTO users (full_name, username, email, password, total_balance) VALUES ($1, $2, $3, $4, $5) RETURNING id',
        [fullName, username, email, hashedPassword, 0]
      );
      const userId = userResult.rows[0].id;
      
      // Create default AUD account
      await client.query(
        'INSERT INTO accounts (user_id, currency, balance) VALUES ($1, $2, $3)',
        [userId, 'AUD', 0]
      );
      
      await client.query('COMMIT');
      return userId;
    } catch (error) {
      await client.query('ROLLBACK');
      throw error;
    } finally {
      client.release();
    }
  }

  async getUserAccounts(userId) {
    await this.initialize();
    const result = await this.pool.query('SELECT * FROM accounts WHERE user_id = $1', [userId]);
    return result.rows;
  }

  async getTotalBalance(userId) {
    await this.initialize();
    const result = await this.pool.query('SELECT total_balance FROM users WHERE id = $1', [userId]);
    return result.rows[0] ? parseFloat(result.rows[0].total_balance) : 0;
  }

  async getAccountBalance(userId, currency) {
    await this.initialize();
    const result = await this.pool.query(
      'SELECT balance FROM accounts WHERE user_id = $1 AND currency = $2',
      [userId, currency]
    );
    return result.rows[0] ? parseFloat(result.rows[0].balance) : 0;
  }

  async updateAccountBalance(userId, currency, amount) {
    await this.initialize();
    const client = await this.pool.connect();
    try {
      await client.query('BEGIN');
      
      // Check if account exists
      const checkResult = await client.query(
        'SELECT id FROM accounts WHERE user_id = $1 AND currency = $2',
        [userId, currency]
      );
      
      if (checkResult.rows.length === 0) {
        // Create account if it doesn't exist
        await client.query(
          'INSERT INTO accounts (user_id, currency, balance) VALUES ($1, $2, $3)',
          [userId, currency, amount]
        );
      } else {
        // Update existing account
        await client.query(
          'UPDATE accounts SET balance = balance + $1 WHERE user_id = $2 AND currency = $3',
          [amount, userId, currency]
        );
      }
      
      // Update users.total_balance to match accounts balance
      const totalResult = await client.query(
        'SELECT SUM(balance) as total FROM accounts WHERE user_id = $1',
        [userId]
      );
      const newTotal = totalResult.rows[0] ? parseFloat(totalResult.rows[0].total) : 0;
      
      // Update users.total_balance
      await client.query(
        'UPDATE users SET total_balance = $1 WHERE id = $2',
        [newTotal, userId]
      );
      
      await client.query('COMMIT');
      return 1;
    } catch (error) {
      await client.query('ROLLBACK');
      throw error;
    } finally {
      client.release();
    }
  }

  async createTransfer(transferData) {
    await this.initialize();
    const result = await this.pool.query(
      `INSERT INTO transfers (
        user_id, from_currency, to_currency, send_amount, fee,
        exchange_rate, receive_amount, recipient_name, recipient_account,
        recipient_bank, recipient_country, rate_locked_at, rate_expires_at, status
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, 'PENDING') RETURNING id`,
      [
        transferData.user_id,
        transferData.from_currency,
        transferData.to_currency,
        transferData.send_amount,
        transferData.fee,
        transferData.exchange_rate,
        transferData.receive_amount,
        transferData.recipient_name,
        transferData.recipient_account,
        transferData.recipient_bank,
        transferData.recipient_country,
        transferData.rate_locked_at,
        transferData.rate_expires_at
      ]
    );
    return result.rows[0].id;
  }

  async getTransferById(transferId) {
    await this.initialize();
    const result = await this.pool.query('SELECT * FROM transfers WHERE id = $1', [transferId]);
    return result.rows[0] || null;
  }

  async updateTransferStatus(transferId, status) {
    await this.initialize();
    const result = await this.pool.query(
      'UPDATE transfers SET status = $1 WHERE id = $2',
      [status, transferId]
    );
    return result.rowCount;
  }

  async getUserTransfers(userId, limit = 50) {
    await this.initialize();
    const limitNum = parseInt(limit, 10);
    const result = await this.pool.query(
      'SELECT * FROM transfers WHERE user_id = $1 ORDER BY created_at DESC LIMIT $2',
      [userId, limitNum]
    );
    return result.rows;
  }

  // Session management methods
  async setSession(sessionId, userId, expiresAt) {
    await this.initialize();
    await this.pool.query(
      `INSERT INTO sessions (session_id, user_id, expires_at) 
       VALUES ($1, $2, $3)
       ON CONFLICT (session_id) 
       DO UPDATE SET user_id = EXCLUDED.user_id, expires_at = EXCLUDED.expires_at`,
      [sessionId, userId, expiresAt]
    );
  }

  async getSession(sessionId) {
    await this.initialize();
    const result = await this.pool.query(
      'SELECT * FROM sessions WHERE session_id = $1 AND expires_at > CURRENT_TIMESTAMP',
      [sessionId]
    );
    return result.rows[0] || null;
  }

  async setTransferData(sessionId, transferData) {
    await this.initialize();
    await this.pool.query(
      'UPDATE sessions SET transfer_data = $1::jsonb WHERE session_id = $2',
      [JSON.stringify(transferData), sessionId]
    );
  }

  async getTransferData(sessionId) {
    await this.initialize();
    const result = await this.pool.query(
      'SELECT transfer_data FROM sessions WHERE session_id = $1 AND expires_at > CURRENT_TIMESTAMP',
      [sessionId]
    );
    if (result.rows[0] && result.rows[0].transfer_data) {
      return result.rows[0].transfer_data;
    }
    return null;
  }

  async clearTransferData(sessionId) {
    await this.initialize();
    await this.pool.query(
      'UPDATE sessions SET transfer_data = NULL WHERE session_id = $1',
      [sessionId]
    );
  }

  async deleteSession(sessionId) {
    await this.initialize();
    await this.pool.query('DELETE FROM sessions WHERE session_id = $1', [sessionId]);
  }

  async cleanupExpiredSessions() {
    await this.initialize();
    await this.pool.query('DELETE FROM sessions WHERE expires_at < CURRENT_TIMESTAMP');
  }
}

module.exports = Database;

