class ExchangeRate {
  constructor() {
    this.API_KEY = '237f6e67ee2389de0cc0e4f5';
    
    this.FALLBACK_RATES = {
      'AUD': {
        'USD': 0.65,
        'EUR': 0.60,
        'GBP': 0.52,
        'NGN': 950,
        'INR': 54,
        'JPY': 97
      },
      'USD': {
        'AUD': 1.54,
        'EUR': 0.93,
        'GBP': 0.79,
        'NGN': 1460,
        'INR': 83,
        'JPY': 149
      },
      'EUR': {
        'AUD': 1.66,
        'USD': 1.07,
        'GBP': 0.85,
        'NGN': 1570,
        'INR': 89,
        'JPY': 160
      },
      'GBP': {
        'AUD': 1.92,
        'USD': 1.27,
        'EUR': 1.18,
        'NGN': 1850,
        'INR': 105,
        'JPY': 189
      }
    };
  }

  async getLiveRate(fromCurrency, toCurrency) {
    // If same currency, return 1
    if (fromCurrency === toCurrency) {
      return 1;
    }

    // Try ExchangeRate API v6
    try {
      const url = `https://v6.exchangerate-api.com/v6/${this.API_KEY}/latest/${fromCurrency}`;
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 5000);

      const response = await fetch(url, {
        method: 'GET',
        signal: controller.signal,
      });

      clearTimeout(timeoutId);

      if (response.ok) {
        const data = await response.json();
        if (data && data.conversion_rates && data.conversion_rates[toCurrency]) {
          return parseFloat(data.conversion_rates[toCurrency]);
        }
      }
    } catch (error) {
      // Fall through to fallback
    }

    // Fallback to hardcoded rates
    if (this.FALLBACK_RATES[fromCurrency] && this.FALLBACK_RATES[fromCurrency][toCurrency]) {
      return this.FALLBACK_RATES[fromCurrency][toCurrency];
    }

    // If no fallback, calculate via USD
    if (fromCurrency !== 'USD' && toCurrency !== 'USD') {
      const toUSD = this.FALLBACK_RATES[fromCurrency]?.USD || 1;
      const fromUSD = 1 / (this.FALLBACK_RATES['USD']?.[toCurrency] || 1);
      return toUSD * fromUSD;
    }

    return 1;
  }

  calculateFee(amount, currency) {
    const feePercentage = 0.007; // 0.7%
    const fee = amount * feePercentage;

    const minFees = {
      'AUD': 0.50,
      'USD': 0.50,
      'EUR': 0.50,
      'GBP': 0.50,
      'NGN': 50,
      'INR': 10
    };

    const minFee = minFees[currency] || 0.50;
    return Math.max(fee, minFee);
  }

  async calculateTransfer(sendAmount, fromCurrency, toCurrency) {
    const exchangeRate = await this.getLiveRate(fromCurrency, toCurrency);
    const fee = this.calculateFee(sendAmount, fromCurrency);
    const amountAfterFee = sendAmount - fee;
    const receiveAmount = Math.round(amountAfterFee * exchangeRate * 100) / 100;

    return {
      sendAmount: sendAmount,
      fee: fee,
      exchangeRate: exchangeRate,
      amountAfterFee: amountAfterFee,
      receiveAmount: receiveAmount
    };
  }
}

module.exports = ExchangeRate;

