const express = require('express');
const cors = require('cors');
const errorHandler = require('./middleware/errorHandler');

// Import routes
const authRoutes = require('./routes/auth');
const dashboardRoutes = require('./routes/dashboard');
const transfersRoutes = require('./routes/transfers');
const lockRateRoutes = require('./routes/transfer/lock-rate');
const recipientRoutes = require('./routes/transfer/recipient');
const confirmRoutes = require('./routes/transfer/confirm');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Request logging (optional, for debugging)
if (process.env.NODE_ENV !== 'production') {
  app.use((req, res, next) => {
    console.log(`${req.method} ${req.path}`);
    next();
  });
}

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({ status: 'ok', timestamp: new Date().toISOString() });
});

// API Routes
app.use('/api', authRoutes);
app.use('/api', dashboardRoutes);
app.use('/api', transfersRoutes);
app.use('/api', lockRateRoutes);
app.use('/api', recipientRoutes);
app.use('/api', confirmRoutes);

// 404 handler
app.use((req, res) => {
  res.status(404).json({
    success: false,
    error: 'Not found',
    message: `Route ${req.method} ${req.path} not found`
  });
});

// Error handling middleware (must be last)
app.use(errorHandler);

// Export for Vercel serverless functions
module.exports = app;

// For local development
if (require.main === module) {
  const { appConfig } = require('./config');
  const port = appConfig.port;
  app.listen(port, () => {
    console.log(`Server running on port ${port}`);
  });
}

