const jwt = require('jsonwebtoken');
const { jwtConfig } = require('../config');
const Database = require('../db');

// JWT-based authentication middleware
const authenticateJWT = async (req, res, next) => {
  try {
    // Get token from Authorization header or cookie
    const authHeader = req.headers.authorization;
    let token = null;

    if (authHeader && authHeader.startsWith('Bearer ')) {
      token = authHeader.substring(7);
    } else if (req.cookies && req.cookies.token) {
      token = req.cookies.token;
    } else if (req.query && req.query.token) {
      token = req.query.token;
    }

    if (!token) {
      return res.status(401).json({
        success: false,
        error: 'Not authenticated',
        message: 'Please log in to continue'
      });
    }

    // Verify JWT token
    const decoded = jwt.verify(token, jwtConfig.secret);
    req.userId = decoded.userId;
    req.user = decoded.user;
    next();
  } catch (error) {
    // If JWT fails, try database session as fallback
    return authenticateSession(req, res, next);
  }
};

// Database session-based authentication middleware
const authenticateSession = async (req, res, next) => {
  try {
    // Get session ID from Authorization header or query param
    const authHeader = req.headers.authorization;
    let sessionId = null;

    if (authHeader) {
      sessionId = authHeader;
    } else if (req.query && req.query.sessionId) {
      sessionId = req.query.sessionId;
    }

    if (!sessionId) {
      return res.status(401).json({
        success: false,
        error: 'Not authenticated',
        message: 'Please log in to continue'
      });
    }

    const db = new Database();
    const session = await db.getSession(sessionId);

    if (!session) {
      return res.status(401).json({
        success: false,
        error: 'Not authenticated',
        message: 'Session expired or invalid'
      });
    }

    req.userId = session.user_id;
    req.sessionId = sessionId;
    next();
  } catch (error) {
    return res.status(401).json({
      success: false,
      error: 'Not authenticated',
      message: 'Authentication failed'
    });
  }
};

// Combined authentication - tries JWT first, then session
const authenticate = async (req, res, next) => {
  // Try JWT first
  const authHeader = req.headers.authorization;
  
  if (authHeader && authHeader.startsWith('Bearer ')) {
    return authenticateJWT(req, res, next);
  }
  
  // Otherwise try session
  return authenticateSession(req, res, next);
};

module.exports = {
  authenticate,
  authenticateJWT,
  authenticateSession
};

