// Centralized error handling middleware
const errorHandler = (err, req, res, next) => {
  console.error('Error:', err);

  // Database errors
  if (err.code && err.code.startsWith('23')) {
    // PostgreSQL constraint violation
    return res.status(400).json({
      success: false,
      error: 'Database constraint violation',
      message: err.message || 'Invalid data provided'
    });
  }

  if (err.code === '23505') {
    // Unique constraint violation
    return res.status(400).json({
      success: false,
      error: 'Duplicate entry',
      message: 'This record already exists'
    });
  }

  // Validation errors
  if (err.name === 'ValidationError') {
    return res.status(400).json({
      success: false,
      error: 'Validation error',
      message: err.message
    });
  }

  // JWT errors
  if (err.name === 'JsonWebTokenError' || err.name === 'TokenExpiredError') {
    return res.status(401).json({
      success: false,
      error: 'Authentication error',
      message: 'Invalid or expired token'
    });
  }

  // Default error response
  const statusCode = err.statusCode || 500;
  const message = err.message || 'An error occurred. Please try again later.';

  res.status(statusCode).json({
    success: false,
    error: err.name || 'Server error',
    message: process.env.NODE_ENV === 'production' 
      ? 'An error occurred. Please try again later.'
      : message
  });
};

module.exports = errorHandler;

