const express = require('express');
const { authenticate } = require('../../middleware/auth');
const Database = require('../../db');
const router = express.Router();

// POST /api/transfer/confirm
router.post('/transfer/confirm', authenticate, async (req, res, next) => {
  try {
    const userId = parseInt(req.userId, 10);
    const sessionId = req.sessionId || req.headers.authorization;

    // Get transfer data from session
    let transferData = null;
    if (sessionId) {
      const db = new Database();
      transferData = await db.getTransferData(sessionId);
    }

    if (!transferData) {
      return res.status(400).json({
        success: false,
        error: 'Transfer data not found',
        message: 'Transfer data not found. Please start a new transfer.'
      });
    }

    const db = new Database();
    const client = await db.pool.connect();

    try {
      await client.query('BEGIN');

      // Check current balance
      const balanceResult = await client.query(
        'SELECT balance FROM accounts WHERE user_id = $1 AND currency = $2',
        [userId, transferData.from_currency]
      );
      const account = balanceResult.rows[0];
      
      const currentBalance = account ? parseFloat(account.balance) : 0;
      const totalToDeduct = transferData.send_amount + transferData.fee;
      
      // Check if user has sufficient balance
      if (currentBalance < totalToDeduct) {
        await client.query('ROLLBACK');
        return res.status(400).json({
          success: false,
          error: 'Insufficient balance',
          message: `You have ${currentBalance} ${transferData.from_currency}, but need ${totalToDeduct} ${transferData.from_currency}`
        });
      }

      // Deduct balance from account
      if (account) {
        await client.query(
          'UPDATE accounts SET balance = balance - $1 WHERE user_id = $2 AND currency = $3',
          [totalToDeduct, userId, transferData.from_currency]
        );
      } else {
        await client.query(
          'INSERT INTO accounts (user_id, currency, balance) VALUES ($1, $2, $3)',
          [userId, transferData.from_currency, -totalToDeduct]
        );
      }

      // Update users.total_balance
      const totalResult = await client.query(
        'SELECT SUM(balance) as total FROM accounts WHERE user_id = $1',
        [userId]
      );
      const newTotal = totalResult.rows[0] ? parseFloat(totalResult.rows[0].total) : 0;
      
      await client.query(
        'UPDATE users SET total_balance = $1 WHERE id = $2',
        [newTotal, userId]
      );

      // Insert transfer record
      const transferResult = await client.query(
        `INSERT INTO transfers (
          user_id, from_currency, to_currency, send_amount, fee,
          exchange_rate, receive_amount, recipient_name, recipient_account,
          recipient_bank, recipient_country, rate_locked_at, rate_expires_at, status
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, 'COMPLETED') RETURNING id`,
        [
          userId,
          transferData.from_currency,
          transferData.to_currency,
          transferData.send_amount,
          transferData.fee,
          transferData.exchange_rate,
          transferData.receive_amount,
          transferData.recipient_name,
          transferData.recipient_account,
          transferData.recipient_bank,
          transferData.recipient_country,
          transferData.rate_locked_at,
          transferData.rate_expires_at
        ]
      );

      const transferId = transferResult.rows[0].id;

      // Commit transaction
      await client.query('COMMIT');

      // Clear transfer data from session
      if (sessionId) {
        await db.clearTransferData(sessionId);
      }

      // Return success
      res.json({
        success: true,
        transfer: {
          id: transferId,
          from_currency: transferData.from_currency,
          to_currency: transferData.to_currency,
          send_amount: transferData.send_amount,
          fee: transferData.fee,
          receive_amount: transferData.receive_amount,
          status: 'COMPLETED'
        }
      });
    } catch (error) {
      await client.query('ROLLBACK');
      throw error;
    } finally {
      client.release();
    }
  } catch (error) {
    next(error);
  }
});

module.exports = router;

