const express = require('express');
const { authenticate } = require('../../middleware/auth');
const ExchangeRate = require('../../exchangeRate');
const router = express.Router();

// POST /api/transfer/lock-rate
router.post('/transfer/lock-rate', authenticate, async (req, res, next) => {
  try {
    const { sendAmount, fromCurrency, toCurrency } = req.body;

    if (!sendAmount || sendAmount <= 0 || !fromCurrency || !toCurrency) {
      return res.status(400).json({
        success: false,
        error: 'Missing required fields',
        message: 'sendAmount, fromCurrency, and toCurrency are required'
      });
    }

    // Simple exchange rate calculation (fallback rates)
    const fallbackRates = {
      'AUD': { 'EUR': 0.60, 'USD': 0.65, 'GBP': 0.52 },
      'USD': { 'EUR': 0.93, 'AUD': 1.54, 'GBP': 0.79 },
      'EUR': { 'USD': 1.07, 'AUD': 1.66, 'GBP': 0.85 },
      'GBP': { 'USD': 1.27, 'EUR': 1.18, 'AUD': 1.92 }
    };

    const rate = fallbackRates[fromCurrency]?.[toCurrency] || 1.0;
    
    // Calculate fees (0.7% fee)
    const fee = sendAmount * 0.007;
    const amountAfterFees = sendAmount - fee;
    const receiveAmount = amountAfterFees * rate;

    // Return transfer details
    const now = new Date();
    const expiresAt = new Date(now.getTime() + 13 * 60 * 60 * 1000); // 13 hours

    res.json({
      success: true,
      sendAmount: sendAmount,
      fee: Math.round(fee * 100) / 100,
      amountAfterFees: Math.round(amountAfterFees * 100) / 100,
      fromCurrency: fromCurrency,
      toCurrency: toCurrency,
      exchangeRate: rate,
      receiveAmount: Math.round(receiveAmount * 100) / 100,
      rateLockedAt: now.toISOString().replace('T', ' ').substring(0, 19),
      rateExpiresAt: expiresAt.toISOString().replace('T', ' ').substring(0, 19)
    });
  } catch (error) {
    next(error);
  }
});

module.exports = router;

