const express = require('express');
const { authenticate } = require('../../middleware/auth');
const Database = require('../../db');
const router = express.Router();

// POST /api/transfer/recipient
router.post('/transfer/recipient', authenticate, async (req, res, next) => {
  try {
    const userId = parseInt(req.userId, 10);
    const sessionId = req.sessionId || req.headers.authorization;

    // Validate required fields
    const required = ['sendAmount', 'fromCurrency', 'toCurrency', 'fee', 'exchangeRate', 
                     'receiveAmount', 'rateExpiresAt', 'recipientName', 'recipientAccount', 
                     'recipientBank', 'recipientCountry'];
    
    for (const field of required) {
      if (!req.body[field] || req.body[field] === '') {
        return res.status(400).json({
          success: false,
          error: 'Missing required field',
          message: `Field '${field}' is required`
        });
      }
    }

    // Store transfer data in database session
    const transferData = {
      user_id: userId,
      from_currency: req.body.fromCurrency,
      to_currency: req.body.toCurrency,
      send_amount: parseFloat(req.body.sendAmount),
      fee: parseFloat(req.body.fee),
      exchange_rate: parseFloat(req.body.exchangeRate),
      receive_amount: parseFloat(req.body.receiveAmount),
      recipient_name: req.body.recipientName.trim(),
      recipient_account: req.body.recipientAccount.trim(),
      recipient_bank: req.body.recipientBank.trim(),
      recipient_country: req.body.recipientCountry.trim(),
      rate_locked_at: new Date().toISOString().replace('T', ' ').substring(0, 19),
      rate_expires_at: req.body.rateExpiresAt
    };

    if (sessionId) {
      const db = new Database();
      await db.setTransferData(sessionId, transferData);
    }

    res.json({
      success: true,
      message: 'Recipient details saved'
    });
  } catch (error) {
    next(error);
  }
});

module.exports = router;

