const Database = require('./db');
const ExchangeRate = require('./exchangeRate');

class TransferService {
  constructor() {
    this.db = new Database();
    this.exchangeRate = new ExchangeRate();
  }

  async lockRate(sendAmount, fromCurrency, toCurrency) {
    const transferDetails = await this.exchangeRate.calculateTransfer(sendAmount, fromCurrency, toCurrency);
    
    const now = new Date();
    const expiresAt = new Date(now.getTime() + 30 * 1000); // 30 seconds from now
    
    return {
      ...transferDetails,
      rateLockedAt: now.toISOString(),
      rateExpiresAt: expiresAt.toISOString()
    };
  }

  isRateValid(rateExpiresAt) {
    if (!rateExpiresAt) return false;
    const expires = new Date(rateExpiresAt);
    const now = new Date();
    return now < expires;
  }

  async executeTransfer(userId, transferData) {
    const fromCurrency = transferData.from_currency;
    const sendAmount = transferData.send_amount;
    const fee = transferData.fee;
    const rateExpiresAt = transferData.rate_expires_at;
    
    // Validate rate is still valid
    if (!this.isRateValid(rateExpiresAt)) {
      throw new Error('Exchange rate has expired. Please start a new transfer.');
    }
    
    // Check balance
    const currentBalance = await this.db.getAccountBalance(userId, fromCurrency);
    const totalNeeded = sendAmount + fee;
    
    if (currentBalance < totalNeeded) {
      throw new Error(`Insufficient balance. You have ${currentBalance} ${fromCurrency}, but need ${totalNeeded} ${fromCurrency}`);
    }
    
    // Deduct balance
    await this.db.updateAccountBalance(userId, fromCurrency, -totalNeeded);
    
    // Create transfer record
    const transferId = await this.db.createTransfer(transferData);
    
    // Update transfer status to COMPLETED
    await this.db.updateTransferStatus(transferId, 'COMPLETED');
    
    // Return transfer record
    return await this.db.getTransferById(transferId);
  }

  async getTransferHistory(userId, limit = 50) {
    return await this.db.getUserTransfers(userId, limit);
  }
}

module.exports = TransferService;

