<?php
require_once __DIR__ . '/db.php';

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function hashPassword($password) {
        return password_hash($password, PASSWORD_DEFAULT);
    }
    
    public function comparePassword($password, $storedPassword) {
        // Plain text comparison (no hashing)
        return $password === $storedPassword;
    }
    
    public function registerUser($fullName, $username, $email, $password) {
        // Check if user exists by username
        if ($username) {
            $existingUser = $this->db->getUserByUsername($username);
            if ($existingUser) {
                throw new Exception('User with this username already exists');
            }
        }
        
        // Check if user exists by email
        $existingUser = $this->db->getUserByEmail($email);
        if ($existingUser) {
            throw new Exception('User with this email already exists');
        }
        
        // Store password as plain text (no hashing)
        $plainPassword = $password;
        
        // Create user
        $userId = $this->db->createUser($fullName, $username, $email, $plainPassword);
        
        // Return user without password
        $user = $this->db->getUserById($userId);
        unset($user['password']);
        return $user;
    }
    
    public function authenticateUser($username, $password) {
        $user = $this->db->getUserByUsername($username);
        
        if (!$user) {
            return null;
        }
        
        $isMatch = $this->comparePassword($password, $user['password']);
        
        if (!$isMatch) {
            return null;
        }
        
        // Return user without password
        unset($user['password']);
        return $user;
    }
    
    public function getUserDashboard($userId) {
        try {
            $user = $this->db->getUserById($userId);
            
            if (!$user) {
                throw new Exception('User not found with ID: ' . $userId);
            }
            
            // Remove password from user data
            unset($user['password']);
            
            $accounts = $this->db->getUserAccounts($userId);
            $totalBalance = $this->db->getTotalBalance($userId);
            $transfers = $this->db->getUserTransfers($userId, 10);
            
            return [
                'user' => [
                    'id' => $user['id'] ?? 0,
                    'full_name' => $user['full_name'] ?? 'User',
                    'username' => $user['username'] ?? '',
                    'email' => $user['email'] ?? ''
                ],
                'accounts' => is_array($accounts) ? $accounts : [],
                'totalBalance' => $totalBalance ? (float)$totalBalance : 0.0,
                'recentTransfers' => is_array($transfers) ? $transfers : [],
                'transfers' => is_array($transfers) ? $transfers : [] // Also include for compatibility
            ];
        } catch (PDOException $e) {
            throw new Exception('Database error in getUserDashboard: ' . $e->getMessage());
        } catch (Exception $e) {
            throw new Exception('Error getting dashboard data: ' . $e->getMessage());
        }
    }
}

