<?php
// Set headers FIRST - before ANY output (critical for InfinityFree)
header('Content-Type: application/json; charset=utf-8');
header('X-Content-Type-Options: nosniff');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

// Get the root directory
$rootDir = dirname(__DIR__);

// Files/directories to ignore
$ignorePatterns = [
    'node_modules',
    '.git',
    '.vscode',
    '.idea',
    'vendor',
    '*.log',
    'database/bank.db',
    'wise.com.zip',
    '.htaccess'
];

// Get all PHP, HTML, JS, CSS files modified in the last 7 days
$files = [];
$cutoffTime = time() - (7 * 24 * 60 * 60); // 7 days ago

function shouldIgnore($path, $ignorePatterns) {
    foreach ($ignorePatterns as $pattern) {
        if (strpos($path, $pattern) !== false) {
            return true;
        }
    }
    return false;
}

function scanDirectory($dir, $rootDir, &$files, $ignorePatterns, $cutoffTime) {
    $items = scandir($dir);
    
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        
        $path = $dir . DIRECTORY_SEPARATOR . $item;
        $relativePath = str_replace($rootDir . DIRECTORY_SEPARATOR, '', $path);
        
        // Skip ignored files
        if (shouldIgnore($relativePath, $ignorePatterns)) {
            continue;
        }
        
        if (is_dir($path)) {
            scanDirectory($path, $rootDir, $files, $ignorePatterns, $cutoffTime);
        } else {
            // Only include relevant file types
            $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
            if (in_array($ext, ['php', 'html', 'js', 'css', 'json', 'md', 'txt', 'sql'])) {
                $modified = filemtime($path);
                if ($modified >= $cutoffTime) {
                    $files[] = [
                        'path' => $relativePath,
                        'modified' => $modified,
                        'size' => filesize($path)
                    ];
                }
            }
        }
    }
}

try {
    scanDirectory($rootDir, $rootDir, $files, $ignorePatterns, $cutoffTime);
    
    // Sort by modification time (newest first)
    usort($files, function($a, $b) {
        return $b['modified'] - $a['modified'];
    });
    
    // Limit to 50 most recent files
    $files = array_slice($files, 0, 50);
    
    echo json_encode([
        'success' => true,
        'files' => $files,
        'count' => count($files)
    ]);
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

