<?php
// Start output buffering IMMEDIATELY to catch any unexpected output
ob_start();

// Enable error reporting (will be sent to error log, not displayed)
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

// Set headers FIRST - before ANY output
header('Content-Type: application/json; charset=utf-8');
header('X-Content-Type-Options: nosniff');

// Start session
session_start();

// Clear any output before sending response
ob_clean();

try {
    // Check if user is logged in via session
    if (!isset($_SESSION['user_id'])) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'error' => 'Not authenticated',
            'message' => 'Please log in to view your balance'
        ]);
        exit;
    }

    $userId = (int) $_SESSION['user_id'];

    // Load database configuration
    require_once __DIR__ . '/db-config.php';

    // Connect to database directly (simpler than Database class)
    $dsn = "pgsql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME;
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ];

    $pdo = new PDO($dsn, DB_USER, defined('DB_PASS') ? DB_PASS : '', $options);

    // Get user balance
    $stmt = $pdo->prepare("SELECT total_balance FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();

    if (!$user) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'User not found',
            'message' => 'User account not found'
        ]);
        exit;
    }

    // Success - return balance
    echo json_encode([
        'success' => true,
        'balance' => (float)($user['total_balance'] ?? 0),
        'totalBalance' => (float)($user['total_balance'] ?? 0) // Also include for compatibility with dash.html
    ]);
    exit;

} catch (PDOException $e) {
    // Database error
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database error',
        'message' => 'Unable to connect to database. Please try again later.'
    ]);
    exit;
} catch (Exception $e) {
    // Any other error
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Server error',
        'message' => 'An error occurred. Please try again later.'
    ]);
    exit;
}
