<?php
/**
 * MySQL Database Class
 * Alternative to SQLite for better hosting compatibility
 */
class DatabaseMySQL {
    public $db;
    private $host;
    private $dbname;
    private $username;
    private $password;
    
    public function __construct() {
        try {
            // MySQL connection settings - UPDATE THESE WITH YOUR CREDENTIALS
            $this->host = 'localhost'; // or your MySQL host
            $this->dbname = 'your_database_name'; // UPDATE THIS
            $this->username = 'your_username'; // UPDATE THIS
            $this->password = 'your_password'; // UPDATE THIS
            
            $dsn = "mysql:host={$this->host};dbname={$this->dbname};charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];
            
            $this->db = new PDO($dsn, $this->username, $this->password, $options);
            $this->initializeDatabase();
        } catch (PDOException $e) {
            throw new Exception('MySQL connection failed: ' . $e->getMessage());
        } catch (Exception $e) {
            throw new Exception('Database initialization failed: ' . $e->getMessage());
        }
    }
    
    private function initializeDatabase() {
        // Users table
        $this->db->exec("CREATE TABLE IF NOT EXISTS users (
            id INT AUTO_INCREMENT PRIMARY KEY,
            full_name VARCHAR(255) NOT NULL,
            username VARCHAR(100) UNIQUE,
            email VARCHAR(255) UNIQUE NOT NULL,
            password VARCHAR(255) NOT NULL,
            total_balance DECIMAL(15,2) DEFAULT 0,
            currency VARCHAR(10) DEFAULT 'AUD',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
        
        // Accounts table
        $this->db->exec("CREATE TABLE IF NOT EXISTS accounts (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            currency VARCHAR(10) NOT NULL,
            balance DECIMAL(15,2) DEFAULT 0,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            UNIQUE KEY unique_user_currency (user_id, currency)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
        
        // Transfers table
        $this->db->exec("CREATE TABLE IF NOT EXISTS transfers (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            from_currency VARCHAR(10),
            to_currency VARCHAR(10),
            send_amount DECIMAL(15,2),
            fee DECIMAL(15,2),
            exchange_rate DECIMAL(20,6),
            receive_amount DECIMAL(15,2),
            status VARCHAR(50) DEFAULT 'PENDING',
            recipient_name VARCHAR(255),
            recipient_account VARCHAR(255),
            recipient_bank VARCHAR(255),
            recipient_country VARCHAR(100),
            rate_locked_at DATETIME,
            rate_expires_at DATETIME,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_status (status),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    }
    
    public function getUserByEmail($email) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->execute([$email]);
        return $stmt->fetch();
    }
    
    public function getUserByUsername($username) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$username]);
        return $stmt->fetch();
    }
    
    public function getUserById($id) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    public function createUser($fullName, $username, $email, $hashedPassword) {
        $this->db->beginTransaction();
        try {
            $stmt = $this->db->prepare("INSERT INTO users (full_name, username, email, password) VALUES (?, ?, ?, ?)");
            $stmt->execute([$fullName, $username, $email, $hashedPassword]);
            $userId = $this->db->lastInsertId();
            
            // Create default AUD account
            $stmt = $this->db->prepare("INSERT INTO accounts (user_id, currency, balance) VALUES (?, ?, ?)");
            $stmt->execute([$userId, 'AUD', 0]);
            
            $this->db->commit();
            return $userId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function getUserAccounts($userId) {
        $stmt = $this->db->prepare("SELECT * FROM accounts WHERE user_id = ?");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
    
    public function getTotalBalance($userId) {
        $stmt = $this->db->prepare("SELECT SUM(balance) as total_balance FROM accounts WHERE user_id = ?");
        $stmt->execute([$userId]);
        $result = $stmt->fetch();
        return $result['total_balance'] ?? 0;
    }
    
    public function getAccountBalance($userId, $currency) {
        $stmt = $this->db->prepare("SELECT balance FROM accounts WHERE user_id = ? AND currency = ?");
        $stmt->execute([$userId, $currency]);
        $result = $stmt->fetch();
        return $result['balance'] ?? 0;
    }
    
    public function updateAccountBalance($userId, $currency, $amount) {
        // First ensure account exists
        $stmt = $this->db->prepare("SELECT id FROM accounts WHERE user_id = ? AND currency = ?");
        $stmt->execute([$userId, $currency]);
        $account = $stmt->fetch();
        
        if (!$account) {
            // Create account if it doesn't exist
            $stmt = $this->db->prepare("INSERT INTO accounts (user_id, currency, balance) VALUES (?, ?, ?)");
            $stmt->execute([$userId, $currency, $amount]);
        } else {
            // Update existing account
            $stmt = $this->db->prepare("UPDATE accounts SET balance = balance + ? WHERE user_id = ? AND currency = ?");
            $stmt->execute([$amount, $userId, $currency]);
        }
        return $stmt->rowCount();
    }
    
    public function createTransfer($transferData) {
        $stmt = $this->db->prepare("INSERT INTO transfers (
            user_id, from_currency, to_currency, send_amount, fee,
            exchange_rate, receive_amount, recipient_name, recipient_account,
            recipient_bank, recipient_country, rate_locked_at, rate_expires_at, status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'PENDING')");
        
        $stmt->execute([
            $transferData['user_id'],
            $transferData['from_currency'],
            $transferData['to_currency'],
            $transferData['send_amount'],
            $transferData['fee'],
            $transferData['exchange_rate'],
            $transferData['receive_amount'],
            $transferData['recipient_name'],
            $transferData['recipient_account'],
            $transferData['recipient_bank'],
            $transferData['recipient_country'],
            $transferData['rate_locked_at'],
            $transferData['rate_expires_at']
        ]);
        
        return $this->db->lastInsertId();
    }
    
    public function updateTransferStatus($transferId, $status) {
        $stmt = $this->db->prepare("UPDATE transfers SET status = ? WHERE id = ?");
        $stmt->execute([$status, $transferId]);
        return $stmt->rowCount();
    }
    
    public function getTransferById($transferId) {
        $stmt = $this->db->prepare("SELECT * FROM transfers WHERE id = ?");
        $stmt->execute([$transferId]);
        return $stmt->fetch();
    }
    
    public function getUserTransfers($userId, $limit = 50) {
        $stmt = $this->db->prepare("SELECT * FROM transfers WHERE user_id = ? ORDER BY created_at DESC LIMIT ?");
        $stmt->execute([$userId, $limit]);
        return $stmt->fetchAll();
    }
}

