<?php
// Load database configuration
require_once __DIR__ . '/db-config.php';

class Database {
    public $db;
    
    public function __construct() {
        // Use PostgreSQL
        if (!defined('DB_TYPE') || DB_TYPE !== 'postgres') {
            throw new Exception('PostgreSQL is required. Please configure api/db-config.php');
        }
        
        $this->initPostgres();
    }
    
    private function initPostgres() {
        try {
            if (!defined('DB_HOST') || !defined('DB_NAME') || !defined('DB_USER')) {
                throw new Exception('PostgreSQL configuration not found. Please check api/db-config.php');
            }
            
            $dsn = "pgsql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ];
            
            $this->db = new PDO($dsn, DB_USER, defined('DB_PASS') ? DB_PASS : '', $options);
            $this->initializeDatabasePostgres();
        } catch (PDOException $e) {
            throw new Exception('PostgreSQL connection failed: ' . $e->getMessage() . '. Please check your database credentials in api/db-config.php');
        } catch (Exception $e) {
            throw new Exception('PostgreSQL initialization failed: ' . $e->getMessage());
        }
    }
    
    private function initializeDatabasePostgres() {
        // Users table
        $this->db->exec("CREATE TABLE IF NOT EXISTS users (
            id SERIAL PRIMARY KEY,
            full_name VARCHAR(255) NOT NULL,
            username VARCHAR(100) UNIQUE,
            email VARCHAR(255) UNIQUE NOT NULL,
            password VARCHAR(255) NOT NULL,
            total_balance DECIMAL(15,2) DEFAULT 0,
            currency VARCHAR(10) DEFAULT 'AUD',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        
        // Create indexes for users table
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_users_email ON users(email)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_users_username ON users(username)");
        
        // Accounts table
        $this->db->exec("CREATE TABLE IF NOT EXISTS accounts (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
            currency VARCHAR(10) NOT NULL,
            balance DECIMAL(15,2) DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, currency)
        )");
        
        // Create indexes for accounts table
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_accounts_user_id ON accounts(user_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_accounts_currency ON accounts(currency)");
        
        // Transfers table
        $this->db->exec("CREATE TABLE IF NOT EXISTS transfers (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
            from_currency VARCHAR(10),
            to_currency VARCHAR(10),
            send_amount DECIMAL(15,2),
            fee DECIMAL(15,2),
            exchange_rate DECIMAL(20,6),
            receive_amount DECIMAL(15,2),
            status VARCHAR(50) DEFAULT 'PENDING',
            recipient_name VARCHAR(255),
            recipient_account VARCHAR(255),
            recipient_bank VARCHAR(255),
            recipient_country VARCHAR(100),
            rate_locked_at TIMESTAMP,
            rate_expires_at TIMESTAMP,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        
        // Create indexes for transfers table
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_transfers_user_id ON transfers(user_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_transfers_status ON transfers(status)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_transfers_created_at ON transfers(created_at)");
        
        // Sessions table for database-backed sessions
        $this->db->exec("CREATE TABLE IF NOT EXISTS sessions (
            session_id VARCHAR(255) PRIMARY KEY,
            user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
            transfer_data JSONB,
            expires_at TIMESTAMP,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        
        // Create index for sessions table
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_sessions_user_id ON sessions(user_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_sessions_expires_at ON sessions(expires_at)");
    }
    
    
    public function getUserByEmail($email) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->execute([$email]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function getUserByUsername($username) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$username]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function getUserById($id) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function createUser($fullName, $username, $email, $hashedPassword) {
        $this->db->beginTransaction();
        try {
            // Create user with initial balance of 0
            $stmt = $this->db->prepare("INSERT INTO users (full_name, username, email, password, total_balance) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$fullName, $username, $email, $hashedPassword, 0]);
            $userId = $this->db->lastInsertId();
            
            // Create default AUD account
            $stmt = $this->db->prepare("INSERT INTO accounts (user_id, currency, balance) VALUES (?, ?, ?)");
            $stmt->execute([$userId, 'AUD', 0]);
            
            $this->db->commit();
            return $userId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function getUserAccounts($userId) {
        $stmt = $this->db->prepare("SELECT * FROM accounts WHERE user_id = ?");
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getTotalBalance($userId) {
        // Get balance from users.total_balance field (not from accounts table)
        $stmt = $this->db->prepare("SELECT total_balance FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['total_balance'] ?? 0;
    }
    
    public function getAccountBalance($userId, $currency) {
        $stmt = $this->db->prepare("SELECT balance FROM accounts WHERE user_id = ? AND currency = ?");
        $stmt->execute([$userId, $currency]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['balance'] ?? 0;
    }
    
    public function updateAccountBalance($userId, $currency, $amount) {
        // Check if account exists
        $checkStmt = $this->db->prepare("SELECT id FROM accounts WHERE user_id = ? AND currency = ?");
        $checkStmt->execute([$userId, $currency]);
        $account = $checkStmt->fetch();
        
        if (!$account) {
            // Create account if it doesn't exist
            $insertStmt = $this->db->prepare("INSERT INTO accounts (user_id, currency, balance) VALUES (?, ?, ?)");
            $insertStmt->execute([$userId, $currency, $amount]);
        } else {
            // Update existing account
            $stmt = $this->db->prepare("UPDATE accounts SET balance = balance + ? WHERE user_id = ? AND currency = ?");
            $stmt->execute([$amount, $userId, $currency]);
        }
        
        // Update users.total_balance to match accounts balance
        $stmt = $this->db->prepare("SELECT SUM(balance) as total FROM accounts WHERE user_id = ?");
        $stmt->execute([$userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $newTotal = $result['total'] ?? 0;
        
        // Update users.total_balance
        $updateUserStmt = $this->db->prepare("UPDATE users SET total_balance = ? WHERE id = ?");
        $updateUserStmt->execute([$newTotal, $userId]);
        
        return 1;
    }
    
    public function createTransfer($transferData) {
        $stmt = $this->db->prepare("INSERT INTO transfers (
            user_id, from_currency, to_currency, send_amount, fee,
            exchange_rate, receive_amount, recipient_name, recipient_account,
            recipient_bank, recipient_country, rate_locked_at, rate_expires_at, status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'PENDING')");
        
        $stmt->execute([
            $transferData['user_id'],
            $transferData['from_currency'],
            $transferData['to_currency'],
            $transferData['send_amount'],
            $transferData['fee'],
            $transferData['exchange_rate'],
            $transferData['receive_amount'],
            $transferData['recipient_name'],
            $transferData['recipient_account'],
            $transferData['recipient_bank'],
            $transferData['recipient_country'],
            $transferData['rate_locked_at'],
            $transferData['rate_expires_at']
        ]);
        
        return $this->db->lastInsertId();
    }
    
    public function getTransferById($transferId) {
        $stmt = $this->db->prepare("SELECT * FROM transfers WHERE id = ?");
        $stmt->execute([$transferId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function updateTransferStatus($transferId, $status) {
        $stmt = $this->db->prepare("UPDATE transfers SET status = ? WHERE id = ?");
        $stmt->execute([$status, $transferId]);
        return $stmt->rowCount();
    }
    
    public function getUserTransfers($userId, $limit = 50) {
        $limit = (int)$limit;
        $stmt = $this->db->prepare("SELECT * FROM transfers WHERE user_id = ? ORDER BY created_at DESC LIMIT " . $limit);
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}

