<?php
require_once __DIR__ . '/db-config.php';

class SessionManager {
    private static $pdo = null;
    
    private static function getConnection() {
        if (self::$pdo === null) {
            try {
                $dsn = "pgsql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME;
                $options = [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ];
                self::$pdo = new PDO($dsn, DB_USER, defined('DB_PASS') ? DB_PASS : '', $options);
            } catch (PDOException $e) {
                throw new Exception('Database connection failed: ' . $e->getMessage());
            }
        }
        return self::$pdo;
    }
    
    public static function setSession($sessionId, $userId, $expiresAt = null) {
        $pdo = self::getConnection();
        
        // Default expiration: 24 hours from now
        if ($expiresAt === null) {
            $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO sessions (session_id, user_id, expires_at) 
            VALUES (?, ?, ?)
            ON CONFLICT (session_id) 
            DO UPDATE SET user_id = EXCLUDED.user_id, expires_at = EXCLUDED.expires_at
        ");
        $stmt->execute([$sessionId, $userId, $expiresAt]);
    }
    
    public static function getUserId($sessionId) {
        $pdo = self::getConnection();
        
        // Clean up expired sessions first
        self::cleanupExpiredSessions();
        
        $stmt = $pdo->prepare("SELECT user_id FROM sessions WHERE session_id = ? AND expires_at > CURRENT_TIMESTAMP");
        $stmt->execute([$sessionId]);
        $result = $stmt->fetch();
        
        return $result ? (int)$result['user_id'] : null;
    }
    
    public static function setTransferData($sessionId, $data) {
        $pdo = self::getConnection();
        
        $jsonData = json_encode($data);
        
        $stmt = $pdo->prepare("
            UPDATE sessions 
            SET transfer_data = CAST(? AS JSONB) 
            WHERE session_id = ?
        ");
        $stmt->execute([$jsonData, $sessionId]);
    }
    
    public static function getTransferData($sessionId) {
        $pdo = self::getConnection();
        
        // Clean up expired sessions first
        self::cleanupExpiredSessions();
        
        $stmt = $pdo->prepare("SELECT transfer_data FROM sessions WHERE session_id = ? AND expires_at > CURRENT_TIMESTAMP");
        $stmt->execute([$sessionId]);
        $result = $stmt->fetch();
        
        if ($result && $result['transfer_data']) {
            return json_decode($result['transfer_data'], true);
        }
        
        return null;
    }
    
    public static function clearTransferData($sessionId) {
        $pdo = self::getConnection();
        
        $stmt = $pdo->prepare("UPDATE sessions SET transfer_data = NULL WHERE session_id = ?");
        $stmt->execute([$sessionId]);
    }
    
    public static function cleanupExpiredSessions() {
        $pdo = self::getConnection();
        
        $stmt = $pdo->prepare("DELETE FROM sessions WHERE expires_at < CURRENT_TIMESTAMP");
        $stmt->execute();
    }
    
    public static function deleteSession($sessionId) {
        $pdo = self::getConnection();
        
        $stmt = $pdo->prepare("DELETE FROM sessions WHERE session_id = ?");
        $stmt->execute([$sessionId]);
    }
}

