<?php
/**
 * Test PostgreSQL Connection
 * Run this file to verify your PostgreSQL database connection
 */
header('Content-Type: text/plain');

echo "Testing PostgreSQL Connection...\n\n";

try {
    require_once __DIR__ . '/db-config.php';
    require_once __DIR__ . '/db.php';
    
    echo "✅ Configuration loaded\n";
    echo "Database Type: " . DB_TYPE . "\n";
    echo "Database Name: " . DB_NAME . "\n";
    echo "Database User: " . DB_USER . "\n";
    echo "Database Host: " . DB_HOST . "\n";
    echo "Database Port: " . DB_PORT . "\n\n";
    
    echo "Connecting to database...\n";
    $db = new Database();
    
    echo "✅ Database connection successful!\n\n";
    
    // Test query
    echo "Testing database query...\n";
    $stmt = $db->db->query("SELECT 1 as test");
    $result = $stmt->fetch();
    echo "✅ Query test successful!\n\n";
    
    // Check if tables exist (PostgreSQL syntax)
    echo "Checking tables...\n";
    $tables = ['users', 'accounts', 'transfers', 'sessions'];
    foreach ($tables as $table) {
        $stmt = $db->db->prepare("
            SELECT EXISTS (
                SELECT FROM information_schema.tables 
                WHERE table_schema = 'public' 
                AND table_name = ?
            )
        ");
        $stmt->execute([$table]);
        $exists = $stmt->fetchColumn();
        
        if ($exists) {
            echo "✅ Table '$table' exists\n";
        } else {
            echo "⚠️  Table '$table' does not exist (will be created on first use)\n";
        }
    }
    
    // Test basic CRUD operations
    echo "\nTesting CRUD operations...\n";
    
    // Test INSERT (using a transaction that will be rolled back)
    $db->db->beginTransaction();
    try {
        // This will be rolled back, so no data is actually inserted
        $stmt = $db->db->prepare("SELECT COUNT(*) FROM users");
        $stmt->execute();
        $userCount = $stmt->fetchColumn();
        echo "✅ SELECT test successful (found $userCount users)\n";
        
        $db->db->rollBack();
    } catch (Exception $e) {
        $db->db->rollBack();
        throw $e;
    }
    
    echo "\n✅ All tests passed! PostgreSQL is ready to use.\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "\nPlease check:\n";
    echo "1. Database credentials in api/db-config.php\n";
    echo "2. Environment variables are set (POSTGRES_HOST, POSTGRES_DATABASE, etc.)\n";
    echo "3. Database exists and is accessible\n";
    echo "4. User has proper permissions\n";
    echo "5. PostgreSQL extension (pdo_pgsql) is installed\n";
    exit(1);
}

