<?php
require_once 'db.php';
require_once 'exchangeRate.php';

class TransferService {
    private $db;
    private $exchangeRate;
    
    public function __construct() {
        $this->db = new Database();
        $this->exchangeRate = new ExchangeRate();
    }
    
    public function lockRate($sendAmount, $fromCurrency, $toCurrency) {
        $transferDetails = $this->exchangeRate->calculateTransfer($sendAmount, $fromCurrency, $toCurrency);
        
        $now = new DateTime();
        $expiresAt = clone $now;
        $expiresAt->modify('+30 seconds');
        
        return array_merge($transferDetails, [
            'rateLockedAt' => $now->format('c'),
            'rateExpiresAt' => $expiresAt->format('c')
        ]);
    }
    
    public function isRateValid($rateExpiresAt) {
        if (!$rateExpiresAt) return false;
        $expires = new DateTime($rateExpiresAt);
        $now = new DateTime();
        return $now < $expires;
    }
    
    public function executeTransfer($userId, $transferData) {
        $fromCurrency = $transferData['from_currency'];
        $sendAmount = $transferData['send_amount'];
        $fee = $transferData['fee'];
        $rateExpiresAt = $transferData['rate_expires_at'];
        
        // Validate rate is still valid
        if (!$this->isRateValid($rateExpiresAt)) {
            throw new Exception('Exchange rate has expired. Please start a new transfer.');
        }
        
        // Check balance
        $currentBalance = $this->db->getAccountBalance($userId, $fromCurrency);
        $totalNeeded = $sendAmount + $fee;
        
        if ($currentBalance < $totalNeeded) {
            throw new Exception("Insufficient balance. You have {$currentBalance} {$fromCurrency}, but need {$totalNeeded} {$fromCurrency}");
        }
        
        // Deduct balance
        $this->db->updateAccountBalance($userId, $fromCurrency, -$totalNeeded);
        
        // Create transfer record
        $transferId = $this->db->createTransfer($transferData);
        
        // Update transfer status to COMPLETED
        $this->db->updateTransferStatus($transferId, 'COMPLETED');
        
        // Return transfer record
        return $this->db->getTransferById($transferId);
    }
    
    public function getTransferHistory($userId, $limit = 50) {
        return $this->db->getUserTransfers($userId, $limit);
    }
}

