// Main Application Logic - Static Website Version
// Uses embedded initial data and localStorage for updates

// ============================================================================
// DATA MANAGEMENT
// ============================================================================

// Initial data embedded directly (no fetch required)
const INITIAL_DATA = {
  "balance": 8899276.66,
  "currency": "AUD",
  "user": {
    "id": 1,
    "username": "shante99",
    "full_name": "Shantelle Margret Dixon",
    "email": "shantelle@wise.com"
  },
  "transfers": [],
  "accounts": [
    {
      "currency": "AUD",
      "balance": 8899276.66
    }
  ]
};

let cachedData = null;

/**
 * Load data from embedded initial data and merge with localStorage
 * @returns {Promise<Object>} Merged data object
 */
async function loadData() {
  try {
    // 1. Use embedded initial data (no fetch needed)
    const initialData = INITIAL_DATA;
    
    // 2. Get updates from localStorage
    const storedDataStr = localStorage.getItem('appData');
    const storedData = storedDataStr ? JSON.parse(storedDataStr) : {};
    
    // 3. Merge: localStorage overrides initial data
    // Ensure balance is always a number
    const storedBalance = storedData.balance !== undefined ? parseFloat(storedData.balance) : null;
    const initialBalance = parseFloat(initialData.balance || 0);
    const finalBalance = !isNaN(storedBalance) ? storedBalance : (!isNaN(initialBalance) ? initialBalance : 0);
    
    const mergedData = {
      ...initialData,
      ...storedData,
      // Ensure balance uses stored value if exists and is valid, otherwise initial
      balance: finalBalance,
      // Merge transfers arrays (localStorage transfers take precedence)
      transfers: storedData.transfers && Array.isArray(storedData.transfers) && storedData.transfers.length > 0 
        ? storedData.transfers 
        : (Array.isArray(initialData.transfers) ? initialData.transfers : []),
      // Keep user from initial data (don't override)
      user: initialData.user || storedData.user,
      // Ensure currency is set
      currency: initialData.currency || storedData.currency || 'AUD'
    };
    
    cachedData = mergedData;
    return mergedData;
  } catch (error) {
    console.error('Error loading data:', error);
    // Fallback to localStorage only
    const storedDataStr = localStorage.getItem('appData');
    if (storedDataStr) {
      try {
        return JSON.parse(storedDataStr);
      } catch (parseError) {
        console.error('Error parsing stored data:', parseError);
      }
    }
    // Last resort: use embedded initial data
    return { ...INITIAL_DATA };
  }
}

/**
 * Save data updates to localStorage
 * @param {Object} updates - Data to update
 */
async function saveData(updates) {
  try {
    // Always load fresh data to ensure we have the complete structure
    // This prevents race conditions where saveData is called before loadData completes
    const currentData = cachedData || await loadData();
    
    // Ensure balance is always a number
    if (updates.balance !== undefined) {
      updates.balance = parseFloat(updates.balance);
      if (isNaN(updates.balance)) {
        throw new Error('Invalid balance value');
      }
    }
    
    // Merge updates while preserving required structure
    const updatedData = {
      ...currentData,
      ...updates,
      // Ensure required fields are preserved
      user: currentData.user || updates.user,
      currency: currentData.currency || updates.currency || 'AUD',
      transfers: updates.transfers !== undefined ? updates.transfers : (currentData.transfers || [])
    };
    
    // Save to localStorage with error handling
    try {
      localStorage.setItem('appData', JSON.stringify(updatedData));
      cachedData = updatedData;
    } catch (storageError) {
      // Handle quota exceeded or other storage errors
      if (storageError.name === 'QuotaExceededError' || storageError.code === 22) {
        console.error('localStorage quota exceeded. Clearing old transfers...');
        // Try to save with only recent transfers (last 50)
        const recentTransfers = updatedData.transfers.slice(-50);
        updatedData.transfers = recentTransfers;
        try {
          localStorage.setItem('appData', JSON.stringify(updatedData));
          cachedData = updatedData;
        } catch (retryError) {
          throw new Error('Storage quota exceeded. Please clear browser data or contact support.');
        }
      } else {
        throw storageError;
      }
    }
  } catch (error) {
    console.error('Error saving data:', error);
    throw error;
  }
}

/**
 * Get current balance (merged from data.json + localStorage)
 * @returns {Promise<number>} Current balance
 */
async function getBalance() {
  const data = await loadData();
  return parseFloat(data.balance || 0);
}

/**
 * Get user information
 * @returns {Promise<Object>} User object
 */
async function getUser() {
  const data = await loadData();
  return data.user || {};
}

// ============================================================================
// AUTHENTICATION
// ============================================================================

/**
 * Check if password matches hardcoded password
 * @param {string} username - Username
 * @param {string} password - Password to check
 * @returns {boolean} True if password matches
 */
function checkPassword(username, password) {
  return username === APP_CONFIG.username && password === APP_CONFIG.password;
}

/**
 * Check if user is logged in
 * @returns {boolean} True if logged in
 */
function isLoggedIn() {
  return localStorage.getItem('isLoggedIn') === 'true';
}

/**
 * Login user
 * @param {string} username - Username
 * @param {string} password - Password
 * @returns {boolean} True if login successful
 */
function login(username, password) {
  if (checkPassword(username, password)) {
    localStorage.setItem('isLoggedIn', 'true');
    localStorage.setItem('username', username);
    return true;
  }
  return false;
}

/**
 * Logout user (clears auth but keeps app data)
 */
function logout() {
  localStorage.removeItem('isLoggedIn');
  localStorage.removeItem('username');
  // Note: Keep appData (balance, transfers) - only clear auth
}

// ============================================================================
// BALANCE OPERATIONS
// ============================================================================

/**
 * Update balance (add or deduct)
 * @param {number} amount - Amount to add (positive) or deduct (negative)
 * @returns {Promise<number>} New balance
 */
async function updateBalance(amount) {
  const data = await loadData();
  const currentBalance = parseFloat(data.balance || 0);
  const amountToAdd = parseFloat(amount);
  
  if (isNaN(currentBalance) || isNaN(amountToAdd)) {
    throw new Error('Invalid balance or amount value');
  }
  
  const newBalance = currentBalance + amountToAdd;
  await saveData({ balance: newBalance });
  return newBalance;
}

/**
 * Get account balance for specific currency
 * @param {string} currency - Currency code (e.g., 'AUD', 'USD')
 * @returns {Promise<number>} Balance for currency
 */
async function getAccountBalance(currency) {
  const data = await loadData();
  if (currency === data.currency) {
    return parseFloat(data.balance || 0);
  }
  // For now, only AUD is supported
  return 0;
}

// ============================================================================
// EXCHANGE RATE OPERATIONS
// ============================================================================

const FALLBACK_RATES = {
  'AUD': { 'USD': 0.65, 'EUR': 0.60, 'GBP': 0.52 },
  'USD': { 'EUR': 0.93, 'AUD': 1.54, 'GBP': 0.79 },
  'EUR': { 'USD': 1.07, 'AUD': 1.66, 'GBP': 0.85 },
  'GBP': { 'USD': 1.27, 'EUR': 1.18, 'AUD': 1.92 }
};

/**
 * Calculate exchange rate between currencies
 * @param {string} fromCurrency - Source currency
 * @param {string} toCurrency - Target currency
 * @returns {number} Exchange rate
 */
function calculateExchangeRate(fromCurrency, toCurrency) {
  if (fromCurrency === toCurrency) return 1;
  return FALLBACK_RATES[fromCurrency]?.[toCurrency] || 1.0;
}

/**
 * Calculate transfer fee
 * @param {number} amount - Transfer amount
 * @param {string} currency - Currency code
 * @returns {number} Fee amount
 */
function calculateFee(amount, currency) {
  const feePercentage = 0.007; // 0.7%
  const fee = amount * feePercentage;
  const minFees = { 'AUD': 0.50, 'USD': 0.50, 'EUR': 0.50, 'GBP': 0.50 };
  return Math.max(fee, minFees[currency] || 0.50);
}

/**
 * Lock exchange rate and calculate transfer details
 * @param {number} sendAmount - Amount to send
 * @param {string} fromCurrency - Source currency
 * @param {string} toCurrency - Target currency
 * @returns {Object} Transfer details with locked rate
 */
function lockExchangeRate(sendAmount, fromCurrency, toCurrency) {
  const rate = calculateExchangeRate(fromCurrency, toCurrency);
  const fee = calculateFee(sendAmount, fromCurrency);
  const amountAfterFees = sendAmount - fee;
  const receiveAmount = amountAfterFees * rate;
  
  const now = new Date();
  const expiresAt = new Date(now.getTime() + 13 * 60 * 60 * 1000); // 13 hours
  
  return {
    sendAmount: parseFloat(sendAmount),
    fee: Math.round(fee * 100) / 100,
    amountAfterFees: Math.round(amountAfterFees * 100) / 100,
    exchangeRate: rate,
    receiveAmount: Math.round(receiveAmount * 100) / 100,
    fromCurrency: fromCurrency,
    toCurrency: toCurrency,
    rateLockedAt: now.toISOString().replace('T', ' ').substring(0, 19),
    rateExpiresAt: expiresAt.toISOString().replace('T', ' ').substring(0, 19)
  };
}

// ============================================================================
// TRANSFER OPERATIONS
// ============================================================================

/**
 * Process and execute a transfer
 * @param {Object} transferData - Transfer data object
 * @returns {Object} Created transfer record
 */
async function processTransfer(transferData) {
  // Validate transfer data
  const validation = validateTransfer(transferData);
  if (!validation.valid) {
    throw new Error(validation.error);
  }
  
  const data = await loadData();
  
  // Ensure balance is a number
  const currentBalance = parseFloat(data.balance || 0);
  if (isNaN(currentBalance)) {
    throw new Error('Invalid balance in data');
  }
  
  // Parse and validate amounts
  const sendAmount = parseFloat(transferData.sendAmount);
  const fee = parseFloat(transferData.fee);
  
  if (isNaN(sendAmount) || isNaN(fee) || sendAmount <= 0 || fee < 0) {
    throw new Error('Invalid send amount or fee');
  }
  
  const totalDeduct = sendAmount + fee;
  
  // Check balance
  if (currentBalance < totalDeduct) {
    throw new Error(`Insufficient balance. You have ${currentBalance.toFixed(2)} ${transferData.fromCurrency}, but need ${totalDeduct.toFixed(2)} ${transferData.fromCurrency}`);
  }
  
  // Create transfer record
  const transfer = {
    id: Date.now(), // Use timestamp as ID
    from_currency: transferData.fromCurrency,
    to_currency: transferData.toCurrency,
    send_amount: sendAmount,
    fee: fee,
    exchange_rate: parseFloat(transferData.exchangeRate) || 1,
    receive_amount: parseFloat(transferData.receiveAmount) || 0,
    recipient_name: transferData.recipientName,
    recipient_account: transferData.recipientAccount,
    recipient_bank: transferData.recipientBank,
    recipient_country: transferData.recipientCountry,
    status: 'COMPLETED',
    created_at: new Date().toISOString()
  };
  
  // Update balance
  const newBalance = currentBalance - totalDeduct;
  
  // Get existing transfers
  const existingTransfers = data.transfers || [];
  
  // Save to localStorage
  await saveData({
    balance: newBalance,
    transfers: [...existingTransfers, transfer]
  });
  
  return transfer;
}

/**
 * Get transfer history
 * @param {number} limit - Maximum number of transfers to return
 * @returns {Promise<Array>} Array of transfer records
 */
async function getTransferHistory(limit = 50) {
  const data = await loadData();
  const transfers = data.transfers || [];
  
  if (!Array.isArray(transfers) || transfers.length === 0) {
    return [];
  }
  
  // Sort by created_at descending (newest first)
  // Handle both ISO strings and timestamp IDs
  const sorted = transfers.sort((a, b) => {
    let dateA, dateB;
    
    if (a.created_at) {
      dateA = new Date(a.created_at);
    } else if (a.id && typeof a.id === 'number') {
      dateA = new Date(a.id);
    } else {
      dateA = new Date(0); // Fallback to epoch
    }
    
    if (b.created_at) {
      dateB = new Date(b.created_at);
    } else if (b.id && typeof b.id === 'number') {
      dateB = new Date(b.id);
    } else {
      dateB = new Date(0); // Fallback to epoch
    }
    
    // Check if dates are valid
    if (isNaN(dateA.getTime())) dateA = new Date(0);
    if (isNaN(dateB.getTime())) dateB = new Date(0);
    
    return dateB.getTime() - dateA.getTime();
  });
  
  const limitNum = parseInt(limit, 10);
  return isNaN(limitNum) ? sorted : sorted.slice(0, limitNum);
}

/**
 * Validate transfer data
 * @param {Object} transferData - Transfer data to validate
 * @returns {Object} Validation result { valid: boolean, error: string }
 */
function validateTransfer(transferData) {
  if (!transferData.sendAmount || transferData.sendAmount <= 0) {
    return { valid: false, error: 'Send amount must be greater than 0' };
  }
  if (!transferData.fromCurrency || !transferData.toCurrency) {
    return { valid: false, error: 'Currencies are required' };
  }
  if (!transferData.recipientName || !transferData.recipientAccount) {
    return { valid: false, error: 'Recipient details are required' };
  }
  return { valid: true };
}

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

/**
 * Format currency amount
 * @param {number} amount - Amount to format
 * @param {string} currency - Currency code
 * @returns {string} Formatted currency string
 */
function formatCurrency(amount, currency = 'AUD') {
  return new Intl.NumberFormat('en-US', {
    style: 'currency',
    currency: currency,
    minimumFractionDigits: 2,
    maximumFractionDigits: 2
  }).format(amount);
}

/**
 * Check if rate is still valid
 * @param {string} rateExpiresAt - ISO date string
 * @returns {boolean} True if rate is still valid
 */
function isRateValid(rateExpiresAt) {
  if (!rateExpiresAt) return false;
  const expires = new Date(rateExpiresAt);
  const now = new Date();
  return now < expires;
}

